﻿#
# Read current measured values example
#
# Requirements: Python 3.6.1 ( www.python.org )
#               P85xx/P86xx device with firmware version 4-5-8-1 or higher
#
#  Purpose of this example is demonstration of communication with P85xx/P86xx device.
#  It is not intended to be used in production environment without further changes.
#

#Connection parameters
TCP_IP = '192.168.1.42'
TCP_PORT = 502

import socket, struct

sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
sock.connect((TCP_IP, TCP_PORT))

try:
    req = struct.pack('12B', 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x03, 0x9C, 0x27, 0x00, 0x05)
    #send ModbusTCP request
    sock.send(req)
    '''
    Modbus TCP request:
       0x00 0x00 0x00 0x00 - transaction ID + protocol ID
       0x00 0x06           - length of following bytes
       0x01                - unit ID
       0x03                - function code (read holding registers)
       0x9C 0x27           - register address (from manual - 0x9C27=channel 1)
       0x00 0x05           - number of registers for reading (four measured values)
    '''

    #read ModbusTCP response
    rcv = sock.recv(64)
    #decode response
    out = struct.unpack(">IHccchhhhh", rcv)
    print("Channel 1:", out[5]/10,
          "\nChannel 2:", out[6]/10,
          "\nChannel 3:", out[7]/10,
          "\nChannel 4:", out[8]/10,
          "\nChannel 5:", out[9]/10)
    '''
    Modbus TCP response:
       0x00 0x00 0x00 0x00 - transaction ID + protocol ID
       0x00 0x0D           - length of following bytes
       0x01                - unit ID
       0x03                - function code (read holding registers)
       0x0A                - bytes count (=5x2)
       0x00 0xEB 0x01 0xDE 0x00 0xEE 0xD5 0x08 0xD5 0x08 - response data
          0x00 0xEB = 235 = 23.5°C
          0x01 0xDE = 478 = 47.8%RH
          0x00 0xEE = 238 = 23.8°C
          0xD5 0x08 = -11000 = n/a
          0xD5 0x08 = -11000 = n/a
    '''
    
finally:
    sock.close()
